import jwt from "jsonwebtoken";
import fs from "fs";
import crypto from "crypto";
import path from "path";
import { fileURLToPath } from "url";

// Get the directory name (ES module equivalent of __dirname)
const __filename = fileURLToPath(import.meta.url);
const __dirname = path.dirname(__filename);

// Load RSA keys (private for signing, public for verification)
const PRIVATE_KEY = fs.readFileSync(
  path.resolve(__dirname, "./keys/private.key"),
  "utf8"
);
const PUBLIC_KEY = fs.readFileSync(
  path.resolve(__dirname, "./keys/public.key"),
  "utf8"
);

/**
 * Generate a secure JWT token.
 * @param {Object} payload - The payload data (e.g., user ID).
 * @param {string} expiresIn - Expiration time (e.g., "1h", "7d").
 * @returns {string} - Signed JWT token.
 */
export function signToken(payload, expiresIn = "1h") {
  return jwt.sign(
    {
      ...payload,
      iat: Math.floor(Date.now() / 1000), // Issued at
      jti: crypto.randomUUID(),
    },
    PRIVATE_KEY,
    {
      algorithm: "RS256",
      expiresIn,
    }
  );
}

/**
 * Verify and decode a JWT token.
 * @param {string} token - The JWT token to verify.
 * @returns {Object|null} - Decoded token payload or null if invalid.
 */
export function verifyToken(token) {
  try {
    return jwt.verify(token, PUBLIC_KEY, { algorithms: ["RS256"] });
  } catch (error) {
    console.error("JWT verification failed:", error.message);
    return null;
  }
}
