import multer from "multer";
import path from "path";
import fs from "fs";
import responseHandler from "../utils/responseHandler.js";
import errorCodes from "../utils/errorCodes.js";

// Define directories
const UPLOADS_DIR = path.join(process.cwd(), "public/uploads");
const CSV_DIR = path.join(process.cwd(), "public/csv");

// Ensure directories exist
[UPLOADS_DIR, CSV_DIR].forEach((dir) => {
  if (!fs.existsSync(dir)) {
    fs.mkdirSync(dir, { recursive: true });
  }
});

// Storage for Profile Picture (Permanent)
const profileStorage = multer.diskStorage({
  destination: UPLOADS_DIR,
  filename: (req, file, cb) => {
    cb(null, `${Date.now()}-${file.originalname}`);
  },
});

// Storage for CSV Files (Can be Temporary)
const csvStorage = multer.diskStorage({
  destination: CSV_DIR,
  filename: (req, file, cb) => {
    cb(null, `${Date.now()}-${file.originalname}`);
  },
});

// File Filters
const fileFilter = (allowedTypes) => (req, file, cb) => {
  if (allowedTypes.includes(file.mimetype)) {
    cb(null, true);
  } else {
    cb(new Error("Invalid file type"), false);
  }
};

// Multer Upload Instances
const uploadProfile = multer({
  storage: profileStorage,
  limits: { fileSize: 0.5 * 1024 * 1024 }, // 2MB limit
  fileFilter: fileFilter(["image/jpeg", "image/png", "image/jpg"]),
});

const uploadCSV = multer({
  storage: csvStorage,
  limits: { fileSize: 5 * 1024 * 1024 }, // 5MB limit
  fileFilter: fileFilter([
    "text/csv",
    "application/vnd.ms-excel",
    "application/vnd.openxmlformats-officedocument.spreadsheetml.sheet",
  ]),
});

// Cleanup Function (For Temporary CSV Files)
const cleanupCSV = (filePath) => {
  setTimeout(() => {
    fs.unlink(filePath, (err) => {
      if (err) console.error("Error deleting file:", err);
    });
  }, 60000); // Deletes after 1 min
};

// Middleware for Handling Uploads
const handleProfileUpload = (req, res, next) => {
  uploadProfile.single("profile_picture")(req, res, (err) => {
    if (err)
      return responseHandler.error(res, errorCodes.FILE_TOO_LARGE, err.message);
    req.file.extention = req.file ? path.extname(req.file.filename) : null;
    next();
  });
};

const handleCSVUpload = (req, res, next) => {
  // console.log(req,"middleware req")
  uploadCSV.single("links_file")(req, res, (err) => {
    if (err)
      return responseHandler.error(res, errorCodes.FILE_TOO_LARGE, err.message);
    req.file.extention = req.file ? path.extname(req.file.filename) : null;
    next();
  });
};

export { handleProfileUpload, handleCSVUpload, cleanupCSV };
