import { PrismaClient as PrismaClient1 } from "../../../prisma/generated/client1/index.js";
import { PrismaClient as PrismaClient2 } from "../../../prisma/generated/client2/index.js";
import bcrypt from "bcrypt";
import signupMail from "../../common/emailTemplate/signUpMail.js";
import responseHandler from "../../utils/responseHandler.js";
import { createToken } from "../../utils/helper.js";
import errorCodes from "../../utils/errorCodes.js";
import { sendMail } from "../../common/sendMail/sendMail.js";
import { signToken } from "../../utils/jwt.js";
import forgotPasswordMail from "../../common/emailTemplate/forgotpassword.js";
import { ROLE_ARRAY } from "../../common/constants/index.js";
import jwt, { decode } from "jsonwebtoken";
import dotenv from "dotenv";
import moment from "moment";
import { customResponse, failureResponse } from "../../utils/responseFn.js";
dotenv.config();
const prismaMasterDb = new PrismaClient1();

const prismaSubMasterDb = new PrismaClient2();

// ORGANISATION REGISTRATIONS CONTROLLER FUNCTION //

const registration = async (req, res) => {
  let userDetails;
  try {
    const {
      first_name,
      last_name,
      email,
      password,
      phone_no,
      country_code,
      country_iso_code,
      condition,
    } = req.body;

    // Check if the user already exists in the subscription database
    const existingUser = await prismaSubMasterDb.User.findUnique({
      where: {
        email: email,
        deleted_at: null,
      },
    });

    if (existingUser) {
      return customResponse(res, false, 400, {}, "Email already taken.");
    }

    // Hash password
    const hashedPassword = await bcrypt.hash(password, 10);

    // Find the "Organisation" role in the master DB //
    const existingRole = await prismaMasterDb.roles.findUnique({
      where: { id: 2 },
    });

    if (!existingRole) {
      return customResponse(res, false, 400, {}, "Role not found.");
    }

    // Create new user in the subscription database (prismaSubMasterDb)
    const newUser = await prismaSubMasterDb.$transaction(async (prisma) => {
      const newUser = await prisma.User.create({
        data: {
          first_name,
          last_name,
          email,
          password: hashedPassword,
          phone_no,
          country_code,
          country_iso_code,
          condition,
          create_by: 0,
        },
      });

      await prisma.UserDetails.create({
        data: {
          user_id: newUser.id,
          organisation_id: newUser.id,
          role_id: 2,
        },
      });

      return newUser; // Return the new user details
    });

    if (!newUser) {
      return customResponse(
        res,
        false,
        400,
        {},
        "User not created please try again."
      );
    }

    let tokenData = {
      user: {
        email: email,
      },
    };
    // Generate token for email verification
    let token = jwt.sign(tokenData, process.env.JWT_SECRET_KEY, {
      expiresIn: "10m",
    });

    const url =
      process.env.NODE_ENV === "production"
        ? `https://${process.env.SERVER_FRONTEND_USER_URL}/verify/${token}`
        : `http://${process.env.LOCAL_FRONTEND_URL}/verify/${token}`;

    const msg = signupMail({ url, name: first_name });

    // Get the permissions for the new user (from master DB)
    const allPermissions = await prismaMasterDb.permissions_list.findUnique({
      where: {
        id: 1,
        deleted_at: null,
      },
    });

    if (!allPermissions) {
      return customResponse(res, false, 400, {}, "Permission not found.");
    }

    try {
      // Send the email for verification
      await sendMail(email, msg.title, msg.description);

      // Fetch subscription details, including features
      const subscriptionDetails = await prismaMasterDb.subscription.findUnique({
        where: {
          id: 1,
          deleted_at: null,
        },
        include: {
          subscription_feature: {
            select: {
              id: true,
              status: true, // Include only the fields you need
              subscription_id: true,
              permission_id: true,
              limit: true,
            },
          },
        },
      });

      if (!subscriptionDetails) {
        return responseHandler.error(
          res,
          errorCodes.BAD_REQUEST,
          "Subscription details not found",
          400
        );
      }

      // Prepare insertedData for user_permissions and user_subscription
      let insertedData = subscriptionDetails.subscription_feature.map(
        (item) => {
          return {
            user_id: newUser.id,
            permission_id: item.permission_id,
            limit: item.limit,
          };
        }
      );
      // Insert data into the user_permissions table in the master DB
      await prismaMasterDb.user_permissions.createMany({
        data: insertedData,
      });

      // Insert data into the user_subscription table in the master DB
      await prismaMasterDb.user_subscription.create({
        data: {
          user_id: newUser.id,
          created_by_id: 1,
          subscription_id: subscriptionDetails.id,
        },
      });

      // Return the success response
      return responseHandler.success(
        res,
        { email, token, userId: newUser.id },
        "Email has been sent successfully. Please verify your email.",
        200
      );
    } catch (error) {
      console.log(error, "Error while sending email or inserting data");
      return responseHandler.error(
        res,
        error,
        "Error during registration",
        400
      );
    }
  } catch (error) {
    console.log(error, "Error during registration process");
    return responseHandler.error(res, error, "Error during registration", 500);
  }
};

// LOGIN //
const login = async (req, res) => {
  console.log("req.body", req.body);

  let data = {};
  try {
    const { email, password } = req.body;

   
    const userDetails = await prismaSubMasterDb.User.findUnique({
      where: { email },
      select: {
        id: true,
        email: true,
        first_name: true,
        last_name: true,
        password: true,
        UserInfo: {
          select: {
            id: true,
            role_id: true,
            user_id: true,
            organisation_id: true,
          },
        },
      },
    });

    if (!userDetails) {
      return customResponse(res, false, 400, {}, "User not exist.");
    }

    const compare = await bcrypt.compare(password, userDetails.password);

    if (!compare) {
      return customResponse(res, false, 400, {}, "Invalid credentials.");
    }
    console.log(userDetails, "userDetails0---->");
    const [userRole, allPermissions] = await Promise.all([
      prismaMasterDb.roles.findUnique({
        where: { id: userDetails?.UserInfo?.role_id, deleted_at: null },
        select: {
          id: true,
          name: true,
        },
      }),
      prismaMasterDb.user_permissions.findMany({
        where: { user_id: userDetails.id, deleted_at: null },
        include: { userPermission: true },
      }),
    ]);

    // console.log(userRole, "role logges");

    let subscriptionPackage = {};
    let package_feature = {};
    if (userRole.id === 1) {
      subscriptionPackage = {
        subscription_feature: allPermissions.map((item) => ({
          permission_id: item.permission_id,
          title: item?.userPermission?.title,
        })),
      };
    } else if (userRole.id === 2) {
      const userSubscriptionsData =
        await prismaMasterDb.user_subscription.findFirst({
          where: { user_id: userDetails.id, deleted_at: null },
          include: {
            usersubscriptions: {
              include: {
                subscription_feature: {
                  include: { features: true },
                },
                package_features: true,
              },
            },
          },
        });

      package_feature =
        userSubscriptionsData?.usersubscriptions?.package_features[0];
      subscriptionPackage = {
        subscriptionName: userSubscriptionsData?.usersubscriptions?.name,
        subcriptionStatus: userSubscriptionsData?.usersubscriptions?.status,
        subscription_feature:
          userSubscriptionsData?.usersubscriptions?.subscription_feature.map(
            (item) => ({
              permission_id: item.permission_id,
              subscription_id: item.subscription_id,
              limit: item.limit,
              title: item.features.title,
            })
          ),
      };
    } else {
      subscriptionPackage = {
        subscription_feature: allPermissions.map((item) => ({
          limit: item.limit,
          permission_id: item.permission_id,
          title: item?.userPermission?.title,
        })),
      };
    }

    let tokenData = {};
    tokenData[userRole.name] = {
      id: userDetails.id,
      email: userDetails.email,
    };

    const {
      id,
      first_name,
      last_name,
      UserInfo: { role_id },
    } = userDetails;

    console.log(role_id, "ROLE ID ---->");
    data = {
      id,
      first_name,
      last_name,
      email,
      subscriptionPackage,
      role_id,
      package_feature,
    };

    data.access_token = signToken(tokenData, "30d");
    console.log(data, "GETTING DATADSADAs");
    return responseHandler.success(res, data, "Login success", 200);
  } catch (err) {
    console.log(err.message, "erereroer");
    return failureResponse(res, err);
  }
};

//verify email //
const verifyEmail = async (req, res) => {
  let data = {};
  try {
    const { token } = req.body;

    let decoded;

    try {
      // console.log(decoded, "andsad", process.JWT_SECRET_KEY);
      decoded = jwt.verify(token, process.env.JWT_SECRET_KEY);
    } catch (error) {
      console.log(error, "innn  thte codeed check");
      return responseHandler.error(
        res,
        errorCodes.BAD_REQUEST,
        "Invalid token or email verification link is expired",
        400
      );
    }

    const email = decoded?.user?.email;

    const existingEntry = await prismaSubMasterDb.User.findUnique({
      where: {
        email: email,
        deleted_at: null,
      },
    });

    if (!existingEntry) {
      return responseHandler.error(
        res,
        errorCodes.NOT_FOUND.code,
        "User not found.",
        400
      );
    }

    // await existingEntry.save();

    return responseHandler.success(
      res,
      {},
      "email verified successfully.",
      200
    );
  } catch (error) {
    return responseHandler.error(
      res,
      errorCodes.BAD_REQUEST,
      error.message,
      500
    );
  }
};

// forgot password //
const forgotPassword = async (req, res) => {
  let data = {};
  try {
    const { email } = req.body;

    const userDetails = await prismaSubMasterDb.User.findUnique({
      where: {
        email: email,
        deleted_at: null,
      },
    });

    if (!userDetails) {
      return responseHandler.error(
        res,
        errorCodes.NOT_FOUND.code,
        "User does not exist",
        400
      );
    }

    const tokenData = {
      user: {
        id: userDetails.id,
        email: userDetails.email,
      },
    };

    const token = jwt.sign(tokenData, process.env.JWT_SECRET_KEY, {
      expiresIn: "10m",
    });

    const msg = forgotPasswordMail({
      url:
        process.env.NODE_ENV === "production"
          ? `https://${process.env.SERVER_FRONTEND_USER_URL}/reset-password/${token}`
          : `http://${process.env.LOCAL_FRONTEND_URL}/reset-password/${token}`,
      name: userDetails.first_name,
    });

    try {
      await sendMail(email, msg.title, msg.description);
    } catch (error) {
      return responseHandler.error(
        res,
        errorCodes.BAD_REQUEST,
        error.message,
        400
      );
    }

    userDetails.verification_token = token;
    // await userDetails.save();

    return responseHandler.success(
      res,
      data,
      "Password reset link sent to your email successfully.",
      200
    );
  } catch (error) {
    console.log(error, "ereoroe incode");
    return responseHandler.error(
      res,
      errorCodes.BAD_REQUEST,
      error.message,
      500
    );
  }
};

// reset password //
const resetPassword = async (req, res) => {
  let data = {};
  try {
    const { token, password } = req.body;

    let decoded;
    try {
      decoded = jwt.verify(token, process.env.JWT_SECRET_KEY);
    } catch (error) {
      return responseHandler.error(
        res,
        errorCodes.BAD_REQUEST,
        error.message,
        400
      );
    }

    const userDetails = await prismaSubMasterDb.User.findUnique({
      where: { id: decoded?.user.id, deleted_at: null },
    });

    if (!userDetails) {
      return responseHandler.error(res, errorCodes.NOT_FOUND, "Erro", 400);
    }

    const hashedPassword = await bcrypt.hash(password, 10);

    const updatedData = await prismaSubMasterDb.User.update({
      where: { id: userDetails?.id },
      data: { password: hashedPassword },
    });
    return responseHandler.success(
      res,
      {},
      "Password updated successfully.",
      200
    );
  } catch (error) {
    console.log(error);
    return responseHandler.error(
      res,
      errorCodes.BAD_REQUEST,
      errorCodes.message,
      500
    );
  }
};

const fetchAllRole = async (req, res) => {
  let data;
  try {
    const userData = req.user;

    const ROLE = Object.getOwnPropertyNames(userData)[0];

    const existUser = await prismaSubMasterDb.User.findUnique({
      where: {
        id: userData[ROLE].id,
        deleted_at: null,
      },
      select: {
        id: true,
        first_name: true,
        last_name: true,
        email: true,
        UserInfo: true,
      },
    });
    if (!existUser) {
      return customResponse(res, false, 400, {}, "User not exists.");
    }

    const fetchRoles = await prismaMasterDb.roles.findUnique({
      where: {
        id: existUser?.UserInfo?.role_id,
        deleted_at: null,
      },
    });

    let fetchAllRoles;
    if (fetchRoles.id == 1) {
      fetchAllRoles = await prismaMasterDb.roles.findMany({
        where: {
          id: {
            not: 1,
          },
          deleted_at: null,
        },
      });
    } else if (fetchRoles.id == 2) {
      fetchAllRoles = await prismaMasterDb.roles.findMany({
        where: {
          id: {
            notIn: [1, 2],
          },
          deleted_at: null,
        },
      });
    } else if (fetchRoles.id == 3) {
      const requiredPermissions = [3, 4, 5];
      const fetchUserPermissions =
        await prismaMasterDb.user_permissions.findMany({
          where: {
            deleted_at: null,
            user_id: existUser.id,
          },
        });

      if (!fetchUserPermissions) {
        return customResponse(
          res,
          false,
          400,
          {},
          "User does not have any permission."
        );
      }

      const matchedPermissionIds = fetchUserPermissions
        .filter((item) => requiredPermissions.includes(item.permission_id))
        .map((item) => item.permission_id);
      console.log(matchedPermissionIds, "MATHECHED PERMISISONS");

      fetchAllRoles = await prismaMasterDb.roles.findMany({
        where: {
          id: {
            in: matchedPermissionIds,
          },
          deleted_at: null,
        },
      });
    } else if (fetchRoles.id === 4) {
      fetchAllRoles = await prismaMasterDb.roles.findMany({
        where: {
          id: {
            notIn: [1, 2, 3, 4],
          },
          deleted_at: null,
        },
      });
    }
    data = fetchAllRoles;
    return responseHandler.success(res, data, "success", 200);
  } catch (err) {
    console.log(err);
    return responseHandler.error(res, err, "Error during fetch role", 500);
  }
};

// create user for project //
const addProjectMembers = async (req, res) => {
  try {
    const userData = req.user;
    const { first_name, last_name, email, password, role_id } = req.body;

    const ROLE = Object.getOwnPropertyNames(userData)[0];

    const existUser = await prismaSubMasterDb.User.findUnique({
      where: {
        id: userData[ROLE].id,
        deleted_at: null,
      },
    });

    if (!existUser) {
      return responseHandler.error(
        res,
        errorCodes.USER_NOT_EXIST,
        errorCodes.USER_NOT_EXIST,
        400
      );
    }

    const existEmail = await prismaSubMasterDb.User.findUnique({
      where: {
        email: email,
        deleted_at: null,
      },
    });

    if (existEmail) {
      return responseHandler.error(
        res,
        errorCodes.EMAIL,
        errorCodes.EMAIL.message,
        400
      );
    }
    let orgId = null;
    if (existUser.role_id === 2) {
      orgId = existUser.id;
    } else if (existUser.role_id === 3) {
      orgId = existUser.create_by;
    }

    // Hash password
    const hashedPassword = await bcrypt.hash(password, 10);
    await prismaSubMasterDb.User.create({
      data: {
        first_name,
        last_name,
        email,
        password: hashedPassword,
        condition: true,
        role_id: Number(role_id),
        create_by: existUser.id,
        is_verified: true,
        status: "ACTIVE",
        organisation_id: orgId,
      },
    });

    return responseHandler.success(res, {}, "succes", 200);
  } catch (err) {
    console.log(err);
    return responseHandler.error(
      res,
      err,
      "Error while add project memeber",
      500
    );
  }
};

// const createTeam = async (req, res) => {
//   try {
//     const userData = req.user;
//     const {
//       first_name,
//       last_name,
//       email,
//       password,
//       role_id = "",
//       assign_to = "",
//     } = req.body;

//     const ROLE = Object.getOwnPropertyNames(userData)[0];

//     const existUser = await prismaSubMasterDb.User.findUnique({
//       where: {
//         id: userData[ROLE].id,
//         deleted_at: null,
//       },
//     });

//     if (!existUser) {
//       return customResponse(res, false, 400, {}, "User not exist.");
//     }

//     const existEmail = await prismaSubMasterDb.User.findUnique({
//       where: {
//         email: email,
//         deleted_at: null,
//       },
//     });

//     if (existEmail) {
//       return customResponse(res, false, 400, {}, "Email already exist.");
//     }

//     // Hash password
//     const hashedPassword = await bcrypt.hash(password, 10);
//     const userCreated = await prismaSubMasterDb.User.create({
//       data: {
//         first_name,
//         last_name,
//         email,
//         password: hashedPassword,
//         condition: true,
//         role_id: Number(role_id),
//         create_by: existUser.id,
//         is_verified: true,
//         status: "ACTIVE",
//         organisation_id: 48,
//       },
//     });

//     let orgId = null;
//     if (existUser.role_id === 2) {
//       orgId = existUser.id;
//     } else if (existUser.role_id === 3 || existUser.role_id === 4) {
//       const findOrg = await prismaSubMasterDb.Team.findUnique({
//         where: {
//           user_id: existUser.id,
//           deleted_at: null,
//         },
//       });
//       orgId = findOrg.organisation_id;
//     }

//     if (!userCreated) {
//       return customResponse(
//         res,
//         false,
//         400,
//         {},
//         "Error while creating new user."
//       );
//     }

//     const addTeam = await prismaSubMasterDb.Team.create({
//       data: {
//         organisation_id: orgId,
//         created_by: existUser.id,
//         user_id: userCreated.id,
//         role_id: existUser?.role_id,
//       },
//     });

//     if (assign_to !== "" && assign_to !== null) {
//       const assignTo = await prismaSubMasterDb.AssignTeamMember.create({
//         data: {
//           assign_to_id: Number(assign_to),
//           assign_by_id: existUser?.id,
//           role_id: existUser?.role_id,
//         },
//       });
//     }

//     return customResponse(res, true, 200, {}, "user created successfully.");
//   } catch (err) {
//     console.log(err);
//     return failureResponse(res, err);
//   }
// };

const createTeam = async (req, res) => {
  try {
    const userData = req.user;
    const {
      first_name,
      last_name,
      email,
      password,
      role_id = "",
      assign_to = "",
      organisationId = "",
    } = req.body;

    const ROLE = Object.getOwnPropertyNames(userData)[0];

    const existUser = await prismaSubMasterDb.User.findUnique({
      where: {
        id: userData[ROLE].id,
        deleted_at: null,
      },
      select: {
        id: true,
        email: true,
        UserInfo: true,
      },
    });

    if (!existUser) {
      return customResponse(res, false, 400, {}, "User not exist.");
    }

    const existEmail = await prismaSubMasterDb.User.findUnique({
      where: {
        email,
        deleted_at: null,
      },
      select: {
        id: true,
        email: true,
      },
    });

    if (existEmail) {
      return customResponse(res, false, 400, {}, "Email already exist.");
    }

    // Find orgId before transaction
    let orgId = null;
    if (existUser.UserInfo?.role_id === 1) {
      orgId = Number(organisationId);
    } else if (existUser.UserInfo?.role_id === 2) {
      orgId = existUser.id;
    } else if (
      existUser.UserInfo?.role_id === 3 ||
      existUser?.UserInfo?.role_id === 4
    ) {
      orgId = existUser?.UserInfo?.organisation_id;
    }

    // Run transaction
    await prismaSubMasterDb.$transaction(async (tx) => {
      const hashedPassword = await bcrypt.hash(password, 10);

      const userCreated = await tx.User.create({
        data: {
          first_name,
          last_name,
          email,
          password: hashedPassword,
          condition: true,
          create_by: existUser.id,
          is_verified: true,
          status: "ACTIVE",
        },
      });

      await tx.UserDetails.create({
        data: {
          user_id: userCreated?.id,
          organisation_id: orgId,
          role_id: Number(role_id),
        },
      });

      await tx.Team.create({
        data: {
          organisation_id: orgId,
          created_by: existUser.id,
          user_id: userCreated.id,
          role_id: Number(role_id),
        },
      });
    });

    return customResponse(res, true, 200, {}, "User created successfully.");
  } catch (err) {
    console.error("Transaction failed:", err);
    return failureResponse(res, err);
  }
};

const fetchProjectMembers = async (req, res) => {
  let data;
  try {
    const userData = req.user;
    const { page_no, number_of_rows, role_id, project_id = "" } = req.query;

    const ROLE = Object.getOwnPropertyNames(userData)[0];

    const existUser = await prismaSubMasterDb.User.findFirst({
      where: {
        id: userData[ROLE].id,
        deleted_at: null,
      },
      select: {
        id: true,
        first_name: true,
        last_name: true,
        email: true,
        UserInfo: {
          select: {
            id: true,
            role_id: true,
            organisation_id: true,
            user_id: true,
          },
        },
      },
    });

    if (!existUser) {
      return customResponse(res, false, 400, {}, "User not found.");
    }

    let allUsers;
    if (existUser.UserInfo?.role_id == 1) {
      const usersWithBacklinkCounts = await prismaSubMasterDb.User.findMany({
        where: {
           UserInfo: {
            role_id: {
              in: [3, 4, 5],
            },
            organisation_id: existUser?.id,
          },
          deleted_at: null,
          // email: {
          //   contains: search_text,
          //   mode: "insensitive", // Optional: for case-insensitive matching
          // },
        },
        orderBy: {
          created_at: "desc",
        },
        include: {
          Backlink: {
            where: {
              deleted_at: null,
            },
          },
        },
     
      });

      allUsers = usersWithBacklinkCounts;
    } else if (existUser?.UserInfo?.role_id == 2) {
      const usersByOrganisation = await prismaSubMasterDb.User.findMany({
        where: {
          // email: {
          //   contains: search_text,
          //   mode: "insensitive",
          // },
          deleted_at: null,
          UserInfo: {
            role_id: {
              in: [3, 4, 5],
            },
            organisation_id: existUser?.id,
          },
        },
        orderBy: {
          created_at: "desc",
        },
        select: {
          id: true,
          first_name: true,
          last_name: true,
          email: true,
          UserInfo: {
            select: {
              role_id: true,
              organisation_id: true,
              user_id: true,
            },
          },
          memberProjects: {
            where: {
              deleted_at: null, // 👈 exclude deleted memberProjects
            },
            select: {
              id: true,
              assign_to: true,
              project_id: true,
            },
          },
        },
      });

      allUsers = usersByOrganisation;
    } else if (existUser?.UserInfo?.role_id == 3) {
      const allUserForManger = await prismaSubMasterDb.User.findMany({
        where: {
      
          deleted_at: null,
          UserInfo: {
            is: {
              role_id: {
                in: [3, 4, 5],
              },
              user_id: {
                not: existUser?.id,
              },
              organisation_id: existUser?.UserInfo?.organisation_id,
            },
          },
        },
        orderBy: {
          created_at: "desc",
        },
        select: {
          id: true,
          first_name: true,
          last_name: true,
          email: true,
          UserInfo: {
            select: {
              role_id: true,
              organisation_id: true,
              user_id: true,
            },
          },
          memberProjects: {
            where: {
              deleted_at: null, // 👈 exclude deleted memberProjects
            },
            select: {
              id: true,
              assign_to: true,
              project_id: true,
            },
          },
        },
        // skip: skip,
        // take: limit,
      });

      allUsers = allUserForManger;
    } else {
      const allUsersForLead = await prismaSubMasterDb.User.findMany({
        where: {
          // email: {
          //   contains: search_text,
          //   mode: "insensitive", // case-insensitive search
          // },

          deleted_at: null,
          UserInfo: {
            is: {
              role_id: {
                in: [5],
              },
              user_id: {
                not: existUser?.id,
              },
              organisation_id: existUser?.UserInfo?.organisation_id,
            },
          },
        },
        orderBy: {
          created_at: "desc",
        },
        select: {
          id: true,
          first_name: true,
          last_name: true,
          email: true,
          UserInfo: {
            select: {
              role_id: true,
              organisation_id: true,
              user_id: true,
            },
          },
          memberProjects: {
            where: {
              deleted_at: null, // 👈 exclude deleted memberProjects
            },
            select: {
              id: true,
              assign_to: true,
              project_id: true,
            },
          },
        },
        // skip: skip,
        // take: limit,
      });

      allUsers = allUsersForLead;

      //   where: {
      //     email: {
      //       contains: search_text,
      //       mode: "insensitive",
      //     },
      //     id: {
      //       not: existUser.id,
      //     },
      //     role_id: {
      //       in: [5],
      //     },
      //     deleted_at: null,
      //   },
      // });
    }

    const allRoles = await prismaMasterDb.roles.findMany({
      where: {
        deleted_at: null,
      },
    });

    const userDetails = allUsers.map((user) => {
      const role = allRoles.find((role) => role.id === user.role_id); // Find the corresponding role name
      return {
        ...user,
        roleName: role ? role.name : null,
      };
    });

    data = userDetails;
    return responseHandler.success(res, data, "succes", 200);
  } catch (err) {
    console.log(err);
    return responseHandler.error(
      res,
      err,
      "Error during fetch project members",
      500
    );
  }
};

// user details //

const userDetails = async (req, res) => {
  try {
    let data;
    const userData = req.user;
    const { id } = req.params;

    const ROLE = Object.getOwnPropertyNames(userData)[0];
    // EXIST AUTH USER //
    const existUser = await prismaSubMasterDb.User.findUnique({
      where: {
        id: userData[ROLE].id,
        deleted_at: null,
      },
    });

    if (!existUser) {
      return responseHandler.error(
        res,
        errorCodes.USER_NOT_EXIST,
        errorCodes.USER_NOT_EXIST,
        400
      );
    }

    // CHECK USER ID FOUND //

    const userDetails = await prismaSubMasterDb.User.findUnique({
      where: {
        id: Number(id),
        deleted_at: null,
      },
      select: {
        id: true, // Include the id
        first_name: true,
        last_name: true, // Include other fields you want
        email: true,
        role_id: true,
        country_code: true,
        country_iso_code: true,
        phone_no: true,
        status: true,
        // Add other fields you want to include, but exclude `password` and `updated_at`
      },
    });

    const userRoles = await prismaMasterDb.roles.findUnique({
      where: {
        id: userDetails.role_id,
        deleted_at: null,
      },
    });

    const userPermission = await prismaMasterDb.user_permissions.findMany({
      where: {
        user_id: userDetails.id,
        deleted_at: null,
      },
      include: {
        userPermission: true,
      },
    });

    if (!userDetails) {
      return responseHandler.error(
        res,
        errorCodes.USER_NOT_EXIST,
        errorCodes.USER_NOT_EXIST,
        400
      );
    }

    let userInfo = {
      id: userDetails.id,
      role_id: userDetails.role_id,
      role_name: userRoles.name,
      permissions: userPermission,
    };

    data = userInfo;

    return responseHandler.success(res, data, "succes", 200);
  } catch (err) {
    console.log(err);
    return responseHandler.error(
      res,
      err,
      "Error during fetch user details",
      500
    );
  }
};

// assign permission //

const assignPermission = async (req, res) => {
  try {
    let data;
    const { permissionData, user_id, role_id = "" } = req.body;
    const userData = req.user;

    const ROLE = Object.getOwnPropertyNames(userData)[0];
    // EXIST AUTH USER //
    const existUser = await prismaSubMasterDb.User.findUnique({
      where: {
        id: userData[ROLE].id,
        deleted_at: null,
      },
    });

    if (!existUser) {
      return responseHandler.error(
        res,
        errorCodes.USER_NOT_EXIST,
        errorCodes.USER_NOT_EXIST,
        400
      );
    }

    // check limit //

    const checkLimit = await prismaMasterDb.user_permissions.findMany({
      where: {
        user_id: existUser.id,
        deleted_at: null,
      },
    });

    // 2. Loop through the permissionData to compare limits
    for (let permission of permissionData) {
      const { permission_id, limit } = permission;

      // Find the corresponding permission in the user_permissions table
      const userPermission = checkLimit.find(
        (item) => item.permission_id === permission_id
      );

      // Check if the permission exists for the user and compare limits
      if (userPermission) {
        // If the limit exceeds the allowed limit, return an error
        if (limit > userPermission.limit) {
          return responseHandler.error(
            res,
            errorCodes.BAD_REQUEST,
            "Don't have enough limit for permission.",
            400
          );
        }
        // else {
        //           // If permission doesn't exist for the user
        //           return responseHandler.error(
        //             res,
        //             errorCodes.BAD_REQUEST,
        //             "No permission found.",
        //             400
        //           );
        //         }
      }
    }

    // // const
    let permissions = permissionData.map((item) => {
      return {
        permission_id: item.permission_id,
        user_id: user_id,
        limit: Number(item.limit),
      };
    });

    const assignedPermissions = await Promise.all(
      permissions.map(async (permission) => {
        // Check if the permission already exists
        const existingPermission =
          await prismaMasterDb.user_permissions.findFirst({
            where: {
              user_id: permission.user_id,
              permission_id: permission.permission_id,
              deleted_at: null,
            },
          });

        // If the permission exists, update it
        if (existingPermission) {
          return await prismaMasterDb.user_permissions.update({
            where: {
              id: existingPermission.id,
              deleted_at: null,
            },
            data: {
              limit: permission.limit,
            },
          });
        }

        // If the permission doesn't exist, create a new one
        return await prismaMasterDb.user_permissions.create({
          data: {
            user_id: permission.user_id,
            permission_id: permission.permission_id,
            limit: permission.limit,
          },
        });
      })
    );

    return responseHandler.success(res, {}, "permission assigned success", 200);
  } catch (err) {
    console.log(err);
    return responseHandler.error(res, errorCodes.code, err.message, 500);
  }
};

// fetch all permissions//
const getAllUserPermissions = async (req, res) => {
  try {
    let data;
    const userData = req.user;
    const ROLE = Object.getOwnPropertyNames(userData)[0];

    const existUser = await prismaSubMasterDb.User.findUnique({
      where: {
        deleted_at: null,
        id: userData[ROLE].id,
      },
    });

    if (!existUser) {
      return responseHandler.error(
        res,
        errorCodes.USER_NOT_EXIST,
        errorCodes.USER_NOT_EXIST,
        400
      );
    }

    const allPermissionFetch = await prismaMasterDb.permissions_list.findMany({
      where: {
        deleted_at: null,

        id: {
          notIn: [1],
        },
      },
    });

    // Calculate total pages

    data = allPermissionFetch;
    return responseHandler.success(res, data, "success", 200);
  } catch (err) {
    console.log(err);
    return responseHandler.error(res, errorCodes.BAD_REQUEST, err.message, 500);
  }
};

// user permisisons //
const userPermission = async (req, res) => {
  let data;
  try {
    const { user_id } = req.params;

    const userData = req.user;
    const ROLE = Object.getOwnPropertyNames(userData)[0];

    const existUser = await prismaSubMasterDb.User.findUnique({
      where: {
        deleted_at: null,
        id: userData[ROLE].id,
      },
    });

    if (!existUser) {
      return responseHandler.error(
        res,
        errorCodes.USER_NOT_EXIST,
        errorCodes.USER_NOT_EXIST,
        400
      );
    }

    const userPermission = await prismaMasterDb.user_permissions.findMany({
      where: {
        deleted_at: null,
        user_id: Number(user_id),
      },
    });
    data = userPermission;
    return responseHandler.success(res, data, "success", 200);
  } catch (err) {
    console.log(err, "errer in code");
    return responseHandler.error(res, errorCodes.BAD_REQUEST, err.message, 500);
  }
};

const myPermission = async (req, res) => {
  let data;
  try {
    const userData = req.user;
    const ROLE = Object.getOwnPropertyNames(userData)[0];

    const existUser = await prismaSubMasterDb.User.findUnique({
      where: {
        deleted_at: null,
        id: userData[ROLE].id,
      },
    });

    if (!existUser) {
      return responseHandler.error(
        res,
        errorCodes.USER_NOT_EXIST,
        errorCodes.USER_NOT_EXIST,
        400
      );
    }

    const userPermission = await prismaMasterDb.user_permissions.findMany({
      where: {
        deleted_at: null,
        user_id: existUser.id,
      },
      include: {
        userPermission: true,
      },
    });
    data = userPermission;
    return responseHandler.success(res, data, "success", 200);
  } catch (err) {
    return responseHandler.error(res, errorCodes.BAD_REQUEST, err.message, 500);
  }
};

// user reports //
const userReports = async (req, res) => {
  try {
    const userData = req.user;
    const {
      user_id = "",
      project_id = "",
      from_date = "",
      to_date = "",
    } = req.query;
    const ROLE = Object.getOwnPropertyNames(userData)[0];

    const existUser = await prismaSubMasterDb.User.findUnique({
      where: {
        deleted_at: null,
        id: userData[ROLE].id,
      },
    });

    if (!existUser) {
      return responseHandler.error(
        res,
        errorCodes.USER_NOT_EXIST,
        errorCodes.USER_NOT_EXIST,
        400
      );
    }

    // find user by id//
    const userFound = await prismaSubMasterDb.User.findUnique({
      where: {
        deleted_at: null,
        id: Number(user_id),
      },
    });

    if (!userFound) {
      return responseHandler.error(
        res,
        errorCodes.USER_NOT_EXIST,
        errorCodes.USER_NOT_EXIST,
        400
      );
    }

    let userDetails;
    let backLinks;
    let filters = {};

    if (project_id !== "") {
      filters.project_id = project_id;
    }

    const dateFilterProvided = from_date !== "" && to_date !== "";
    if (dateFilterProvided) {
      filters.created_at = {
        gte: new Date(from_date),
        lte: new Date(to_date),
      };
    }

    userDetails = await prismaSubMasterDb.User.findUnique({
      where: { id: userFound.id },
      include: {
        backlinks: {
          where: {
            uploaded_by_id: userFound.id,
            ...filters, // This applies project and date filters if available
          },
        },
        ReferDomain: {
          where: {
            deleted_at: null,
            user_id: userFound.id,
          },
        },
        memberProjects: {
          where: {
            deleted_at: null,
            assign_to: userFound.id,
          },
          select: {
            id: true,
            assign_by: true,
            assign_to: true,
            userProjects: true,
          },
        },
      },
    });

    // Calculate counts from the filtered backlinks
    const totalBacklinksCount = userDetails.backlinks.length;
    const duplicateBacklinks = userDetails.backlinks.filter(
      (link) => link.refering_domain || link.backlink_duplicate
    );
    const duplicateCount = duplicateBacklinks.length;

    let dailyLinksCount;
    if (dateFilterProvided) {
      // When a date filter is provided, all backlinks are within that range.
      // Thus, for "today" (or "this week", etc.) the count is simply the filtered count.
      dailyLinksCount = totalBacklinksCount;
    } else {
      // No date filter provided: Compute dailyLinks as the count for the current day
      const startOfToday = new Date();
      startOfToday.setHours(0, 0, 0, 0);
      const startOfTomorrow = new Date(startOfToday);
      startOfTomorrow.setDate(startOfToday.getDate() + 1);
      dailyLinksCount = userDetails.backlinks.filter((link) => {
        const createdAt = new Date(link.created_at);
        return createdAt >= startOfToday && createdAt < startOfTomorrow;
      }).length;
    }

    const todayStart = new Date();
    todayStart.setHours(0, 0, 0, 0); // Set to the start of the day

    const todayEnd = new Date();
    todayEnd.setHours(23, 59, 59, 999); // Set to the end of the day
    const dailyRd = await prismaSubMasterDb.ReferDomain.count({
      where: {
        user_id: existUser.id,
        deleted_at: null,
        created_at: {
          gte: todayStart, // Greater than or equal to the start of today
          lte: todayEnd, // Less than or equal to the end of today
        },
      },
    });

    const totalReferDomainCount = userDetails.ReferDomain.length;
    const userProjects = userDetails.memberProjects;

    const data = {
      dalyLinks: dailyLinksCount,
      totalLinks: totalBacklinksCount,
      duplicateLinks: duplicateCount,
      totalReferDomain: totalReferDomainCount,
      userProjects: userProjects,
      dailyRdCounts: dailyRd,
    };
    return responseHandler.success(res, data, "success", 200);
  } catch (err) {
    console.log(err);
    return responseHandler.error(
      res,
      errorCodes.BAD_REQUEST,
      errorCodes.BAD_REQUEST.message,
      500
    );
  }
};

// user report links -----//
const userReportLinks = async (req, res) => {
  try {
    const userData = req.user;

    const {
      id = "",
      page_no = 1,
      rows = 10,
      user_id = "",
      project_id = "",
      status = "",
      duplicate = "",
      from_date = "",
      to_date = "",
      anchor_text = "",
      target_url = "",
      index = "",
    } = req.query;

    const ROLE = Object.getOwnPropertyNames(userData)[0];

    const existUser = await prismaSubMasterDb.User.findUnique({
      where: {
        deleted_at: null,
        id: userData[ROLE].id,
      },
    });

    if (!existUser) {
      return responseHandler.error(
        res,
        errorCodes.USER_NOT_EXIST,
        errorCodes.USER_NOT_EXIST,
        400
      );
    }

    const currentPage = page_no ? parseInt(page_no, 10) : 1;
    const limit = rows ? parseInt(rows, 10) : 10;
    const skip = (currentPage - 1) * limit;

    const userFound = await prismaSubMasterDb.User.findUnique({
      where: {
        deleted_at: null,
        id: Number(user_id),
      },
    });

    if (!userFound) {
      return responseHandler.error(
        res,
        errorCodes.USER_NOT_EXIST,
        errorCodes.USER_NOT_EXIST,
        400
      );
    }

    // FILTERS //
    let filter = {};

    if (project_id !== "") {
      filter.project_id = Number(project_id);
    }

    if (status !== "") {
      filter.link_status = Number(status);
    }

    if (duplicate !== "") {
      if (duplicate === "domain") {
        filter.backlink_duplicate = true;
      } else {
        filter.refering_domain = true;
      }
    }

    if (from_date !== "" && to_date !== "") {
      filter.created_at = {
        gte: new Date(from_date), // Greater than or equal to from_date
        lte: new Date(to_date), // Less than or equal to to_date
      };
    }

    if (anchor_text !== "") {
      filter.anchor_text = anchor_text;
    }

    if (target_url !== "") {
      filter.targeted_url = target_url;
    }

    if (index !== "") {
      filter.is_index = index === "true" ? true : false;
    }
    // FILTERS END //

    let userLinks;
    if (userFound.role_id === 1) {
    } else if (userFound.role_id === 2) {
    } else if (userFound.role_id === 3) {
      userLinks = await prismaSubMasterDb.backlink.findMany({
        where: {
          deleted_at: null,
          managed_by: existUser.id,
        },
        skip: skip,
        take: limit,
      });
    } else if (userFound.role_id === 4) {
      userLinks = await prismaSubMasterDb.backlink.findMany({
        where: {
          deleted_at: null,
          managed_by: existUser.id,
        },
        skip: skip,
        take: limit,
      });
    } else {
      userLinks = await prismaSubMasterDb.backlink.findMany({
        where: {
          deleted_at: null,
          uploaded_by_id: userFound.id,
          ...filter,
          // project_id: Number(project_id),
        },
        include: {
          ManagedBy: true,
          created_by: true,
        },
        skip: skip,
        take: limit,
      });
    }

    // Get the total count of backlinks matching the query
    const totalCount = await prismaSubMasterDb.backlink.count({
      where: {
        deleted_at: null,
        // project_id: Number(project_id),
        uploaded_by_id: userFound.id,
      },
    });

    // Calculate total pages
    const totalPages = Math.ceil(totalCount / limit);
    const data = {
      user_links: userLinks,
      totalCounts: totalCount,
      totalPages: totalPages,
      currentPage: currentPage,
    };

    return responseHandler.success(res, data, "success", 200);
  } catch (error) {
    console.log(error);
    return responseHandler.error(
      res,
      errorCodes.BAD_REQUEST,
      errorCodes.BAD_REQUEST.message,
      500
    );
  }
};

// user-detail//

// const userDetails = async () => {
//   try {
//     const userData = req.user;
//     const { id } = req.params;

//     const ROLE = Object.getOwnPropertyNames(userData)[0];

//     const existUser = await prismaSubMasterDb.User.findUnique({
//       where: {
//         id: userData[ROLE].id,
//         deleted_at: null,
//       },
//       select: {
//         id: true,
//         email: email,
//       },
//     });

//     if (!existUser) {
//       return responseHandler.error(
//         res,
//         errorCodes.USER_NOT_EXIST,
//         errorCodes.USER_NOT_EXIST,
//         400
//       );
//     }

//     // const userDetails = await prismaSubMasterDb.User.findUnique({
//     //   where: {
//     //     id: id,
//     //     deleted_at: null,
//     //   },
//     //   select: {
//     //     id: true,
//     //     email: true,
//     //   },
//     // });
//   } catch (err) {
//     return responseHandler.error(
//       res,
//       errorCodes.BAD_REQUEST,
//       errorCodes.BAD_REQUEST.message,
//       500
//     );
//   }
// };

// update user //
const updateUser = async (req, res) => {
  try {
    const userData = req.user;
    const { id } = req.params;

    const ROLE = Object.getOwnPropertyNames(userData)[0];

    const existUser = await prismaSubMasterDb.User.findUnique({
      where: {
        id: userData[ROLE].id,
        deleted_at: null,
      },
    });

    if (!existUser) {
      return responseHandler.error(
        res,
        errorCodes.USER_NOT_EXIST,
        errorCodes.USER_NOT_EXIST,
        400
      );
    }
  } catch (err) {
    return responseHandler.error(res, errorCodes.BAD_REQUEST, err.message, 500);
  }
};

//upload refering domain //

const uploadRd = async (req, res) => {
  try {
    let data = {};

    const userData = req.user;
    const { category_name, rd_links, category_id } = req.body;

    const ROLE = Object.getOwnPropertyNames(userData)[0];

    const existUser = await prismaSubMasterDb.User.findUnique({
      where: {
        id: userData[ROLE].id,
        deleted_at: null,
      },
    });

    if (!existUser) {
      return responseHandler.error(
        res,
        errorCodes.USER_NOT_EXIST,
        errorCodes.USER_NOT_EXIST,
        400
      );
    }

    // finc category //
    let findCategory = await prismaSubMasterDb.ReferDomainCategory.findFirst({
      where: {
        deleted_at: null,
        id: Number(category_id),
        category_name: category_name,
      },
    });

    // if rd category not exist create new category //
    if (!findCategory) {
      findCategory = await prismaSubMasterDb.ReferDomainCategory.create({
        data: {
          category_name: category_name,
          user_id: existUser.id,
        },
      });
    }

    // Query the database for any records that already have one of these rd_link values
    const existingRDs = await prismaSubMasterDb.referDomain.findMany({
      where: {
        rd_link: { in: rd_links },
      },
    });

    const duplicateLinks = existingRDs.map((record) => record.rd_link);

    // Filter out links that are duplicates (i.e., already in the database)
    const newLinks = rd_links.filter((link) => !duplicateLinks.includes(link));

    const allLinks = newLinks.map((item, index) => {
      return {
        rd_link: item,
        category_id: findCategory.id,
        user_id: existUser.id,
      };
    });

    let createRd;
    if (allLinks.length > 0) {
      createRd = await prismaSubMasterDb.referDomain.createMany({
        data: allLinks,
      });
    } else {
      return responseHandler.error(
        res,
        errorCodes.BAD_REQUEST,
        "Links already exists.",
        400
      );
    }

    data = {
      insertedCount: createRd.count,
      duplicates: duplicateLinks,
    };

    return responseHandler.success(res, data, "success", 200);
  } catch (err) {
    console.log(err);
    return responseHandler.error(res, errorCodes.BAD_REQUEST, err.message, 500);
  }
};

const userProjects = async (req, res) => {
  try {
    let data = {};

    const userData = req.user;
    const { page_no = 1, rows = 20, user_id = "" } = req.query;

    const ROLE = Object.getOwnPropertyNames(userData)[0];

    const existUser = await prismaSubMasterDb.User.findUnique({
      where: {
        id: userData[ROLE].id,
        deleted_at: null,
      },
      select: {
        id: true,
        email: true,
        role_id: true,
      },
    });

    if (!existUser) {
      return responseHandler.error(
        res,
        errorCodes.USER_NOT_EXIST,
        errorCodes.USER_NOT_EXIST,
        400
      );
    }

    const pageInt = parseInt(page_no, 10);
    const limitInt = parseInt(rows, 10);

    if (pageInt < 1 || limitInt < 1) {
      return responseHandler.error(res, "Invalid pagination parameters", 400);
    }

    // Calculate skip based on the current page
    const skip = (pageInt - 1) * limitInt;

    const findUser = await prismaSubMasterDb.User.findUnique({
      where: {
        deleted_at: null,
        id: Number(user_id),
      },
      select: {
        id: true,
        role_id: true,
        email: true,
      },
    });

    if (!findUser) {
      return responseHandler.error(
        res,
        errorCodes.USER_NOT_EXIST,
        errorCodes.USER_NOT_EXIST,
        400
      );
    }

    let projects;

    let totalLinksCount;
    let dailyLinkCount;
    let projectData;
    let duplicateLinkCount;
    const startOfToday = new Date();
    startOfToday.setHours(0, 0, 0, 0);
    const startOfTomorrow = new Date(startOfToday);
    startOfTomorrow.setDate(startOfToday.getDate() + 1);
    if (findUser.role_id === 5) {
      projects = await prismaSubMasterDb.ProjectUser.findMany({
        where: {
          deleted_at: null,
          assign_to: findUser.id,
        },
        skip: skip,
        take: limitInt,
        include: {
          userProjects: {
            include: {
              backlinks: {
                where: {
                  deleted_at: null,
                  uploaded_by_id: findUser.id,
                },
              },
            },
          },
        },
      });

      // Calculate today's date range
      // const startOfToday = new Date();
      // startOfToday.setHours(0, 0, 0, 0);
      // const startOfTomorrow = new Date(startOfToday);
      // startOfTomorrow.setDate(startOfToday.getDate() + 1);

      // Loop through the projects and add both totalLinksCount and todayLinksCount
      projects = projects.map((project) => {
        const allBacklinks = project.userProjects.backlinks || [];
        const totalLinksCount = allBacklinks.length;
        const todayLinksCount = allBacklinks.filter((link) => {
          const createdAt = new Date(link.created_at);
          return createdAt >= startOfToday && createdAt < startOfTomorrow;
        }).length;
        return {
          ...project,
          totalLinksCount,
          todayLinksCount,
        };
      });
    }

    let projectsCounts = await prismaSubMasterDb.ProjectUser.count({
      where: {
        deleted_at: null,
        assign_to: findUser.id,
      },
    });
    // Calculate total pages
    const totalPages = Math.ceil(projectsCounts / limitInt);

    data = {
      allProjects: projects,
      totalPages: totalPages,
      totalCounts: projectsCounts,
      limit: limitInt,
      currentPage: page_no,
    };

    return responseHandler.success(res, data, "success", 200);
  } catch (err) {
    console.log(err);
    return responseHandler.error(res, errorCodes.BAD_REQUEST, err.message, 500);
  }
};

const userRd = async (req, res) => {
  try {
    let data = {};

    const userData = req.user;
    const { page_no = 1, rows = 20, user_id = "" } = req.query;

    const ROLE = Object.getOwnPropertyNames(userData)[0];

    const existUser = await prismaSubMasterDb.User.findUnique({
      where: {
        id: userData[ROLE].id,
        deleted_at: null,
      },
      select: {
        id: true,
        email: true,
        role_id: true,
      },
    });

    if (!existUser) {
      return responseHandler.error(
        res,
        errorCodes.USER_NOT_EXIST,
        errorCodes.USER_NOT_EXIST,
        400
      );
    }

    const pageInt = parseInt(page_no, 10);
    const limitInt = parseInt(rows, 10);

    if (pageInt < 1 || limitInt < 1) {
      return responseHandler.error(res, "Invalid pagination parameters", 400);
    }

    // Calculate skip based on the current page
    const skip = (pageInt - 1) * limitInt;

    const fetchUser = await prismaSubMasterDb.User.findUnique({
      where: {
        deleted_at: null,
        id: Number(user_id),
      },
    });

    if (!fetchUser) {
      return responseHandler.error(
        res,
        errorCodes.USER_NOT_EXIST,
        errorCodes.USER_NOT_EXIST,
        400
      );
    }

    const fetchUSerRd = await prismaSubMasterDb.ReferDomain.findMany({
      where: {
        user_id: fetchUser.id,
        deleted_at: null,
      },
      include: {
        RdCategory: true,
      },
      skip: skip,
      take: limitInt,
    });

    const fetchUSerRdCounts = await prismaSubMasterDb.ReferDomain.count({
      where: {
        id: fetchUser.id,
        deleted_at: null,
      },
    });

    const totalPages = Math.ceil(fetchUSerRdCounts / limitInt);

    data = {
      referDomains: fetchUSerRd,
      totalPages: totalPages,
      totalCounts: fetchUSerRdCounts,
      limit: limitInt,
      currentPage: page_no,
    };
    return responseHandler.success(res, data, "success", 200);
  } catch (err) {
    console.log(err, "err logs");
    return responseHandler.error(res, errorCodes.BAD_REQUEST, err.message, 500);
  }
};

// user detail reports //
const userDetailReport = async (req, res) => {
  try {
    let data = {};
    const userData = req.user;

    const { page_no, rows, user_id } = req.query;

    const pageInt = parseInt(page_no, 10);
    const limitInt = parseInt(rows, 10);

    if (pageInt < 1 || limitInt < 1) {
      return responseHandler.error(res, "Invalid pagination parameters", 400);
    }

    const skip = (pageInt - 1) * limitInt;

    const ROLE = Object.getOwnPropertyNames(userData)[0];
    // EXIST AUTH USER //
    const existUser = await prismaSubMasterDb.User.findUnique({
      where: {
        id: userData[ROLE].id,
        deleted_at: null,
      },
      select: {
        id: true,
        email: true,
        UserInfo: {
          select: {
            id: true,
            user_id: true,
            organisation_id: true,
            role_id: true,
          },
        },
      },
    });

    if (!existUser) {
      return customResponse(res, false, 400, {}, "User not found.");
    }

    const userDetails = await prismaSubMasterDb.User.findFirst({
      where: {
        id: Number(user_id),
        deleted_at: null,
      },
      select: {
        id: true,
        email: true,
        first_name: true,
        last_name: true,
        UserInfo: {
          select: {
            id: true,
            user_id: true,
            organisation_id: true,
            role_id: true,
          },
        },

        memberProjects: {
          select: {
            id: true,
            assign_by: true,
            assign_to: true,
            project_id: true,
            deleted_at: true,
            userProjects: {
              select: {
                id: true,
                domain_name: true,
                domain_url: true,
              },
            },
          },
        },
        assignedProjects: {
          select: {
            id: true,
            assign_by: true,
            assign_to: true,
            project_id: true,
            deleted_at: true,
          },
        },
      },
    });

    const totalProjects = userDetails?.memberProjects?.length || 0;
    const totalMembers = userDetails?.assignedProjects?.length || 0;
    data = {
      totalProjectsCounts: totalProjects,
      totalMembers: totalMembers,
      reportDetails: userDetails,
    };

    return responseHandler.success(res, data, "success", 200);
  } catch (err) {
    console.log(err, "ERR in your code");
    return responseHandler.error(res, errorCodes.BAD_REQUEST, err.message, 500);
  }
};

// all user list//

const allUserList = async (req, res) => {
  let data;
  try {
    const userData = req.user;
    const {
      page_no = 1,
      number_of_rows = 20,
      role_id = "",
      search_text = "",
    } = req.query;

    const ROLE = Object.getOwnPropertyNames(userData)[0];

    const existUser = await prismaSubMasterDb.User.findUnique({
      where: {
        id: userData[ROLE].id,
        deleted_at: null,
      },
    });

    if (!existUser) {
      return responseHandler.error(
        res,
        errorCodes.USER_NOT_EXIST,
        errorCodes.USER_NOT_EXIST,
        400
      );
    }

    const page = parseInt(page_no, 10);
    const limit = parseInt(number_of_rows, 10);
    const skip = (page - 1) * limit; // Skip calculation for pagination
    let allUsers;
    const todayStart = moment().startOf("day").toDate();
    const todayEnd = moment().endOf("day").toDate();
    if (existUser.role_id == 1) {
      const usersWithBacklinkCounts = await prismaSubMasterDb.User.findMany({
        where: {
          role_id: {
            notIn: [1, 2],
          },
          deleted_at: null,
          email: {
            contains: search_text,
            mode: "insensitive", // Optional: for case-insensitive matching
          },
        },
        orderBy: {
          created_at: "desc",
        },
        include: {
          Backlink: {
            where: {
              deleted_at: null,
            },
          },
        },
        skip: skip,
        take: limit,
      });
      allUsers = usersWithBacklinkCounts.map((user) => {
        const totalBacklinks = user.Backlink.length;
        const todayBacklinks = user.Backlink.filter((backlink) => {
          const createdAt = new Date(backlink.created_at);
          return createdAt >= todayStart && createdAt <= todayEnd;
        }).length;

        return {
          ...user,
          totalBacklinks,
          todayBacklinks,
        };
      });
    } else if (existUser.role_id == 2) {
      // const usersWithBacklinkCounts = await prismaSubMasterDb.User.findMany({
      //   where: {
      //     AND: [
      //       {
      //         role_id: {
      //           notIn: [1, 2],
      //         },
      //       },
      //       {
      //         OR: [
      //           { organisation_id: existUser.id },
      //           { create_by: existUser.id }, // Match if created_by is equal to existUser.id
      //         ],
      //       },
      //     ],
      //     deleted_at: null,

      //     email: {
      //       contains: search_text,
      //       mode: "insensitive", // Optional: for case-insensitive matching
      //     },
      //   },

      //   include: {
      //     backlinks: {
      //       where: {
      //         deleted_at: null,
      //       },
      //     },
      //   },
      //   orderBy: {
      //     created_at: "desc",
      //   },
      //   skip: skip,
      //   take: limit,
      // });

      // Fetch users based on organisation_id and create_by
      const usersByOrganisation = await prismaSubMasterDb.User.findMany({
        where: {
          AND: [
            {
              role_id: {
                notIn: [1, 2], // Exclude roles 1 and 2
              },
            },
            {
              organisation_id: existUser.id, // Match if organisation_id is equal to existUser.id
            },
          ],
          deleted_at: null, // Ensure deleted_at is null
          email: {
            contains: search_text, // Case-insensitive search by email
            mode: "insensitive",
          },
        },
        include: {
          backlinks: {
            where: {
              deleted_at: null, // Only include backlinks that are not deleted
            },
          },
        },
        orderBy: {
          created_at: "desc", // Order by creation date in descending order
        },
        skip: skip,
        take: limit,
      });

      const usersByCreator = await prismaSubMasterDb.User.findMany({
        where: {
          AND: [
            {
              role_id: {
                notIn: [1, 2], // Exclude roles 1 and 2
              },
            },
            {
              create_by: existUser.id, // Match if create_by is equal to existUser.id
            },
          ],
          deleted_at: null, // Ensure deleted_at is null
          email: {
            contains: search_text, // Case-insensitive search by email
            mode: "insensitive",
          },
        },
        include: {
          backlinks: {
            where: {
              deleted_at: null, // Only include backlinks that are not deleted
            },
          },
        },
        orderBy: {
          created_at: "desc", // Order by creation date in descending order
        },
        skip: skip,
        take: limit,
      });

      // Combine both results
      const combinedResults = [...usersByOrganisation, ...usersByCreator];

      // Remove duplicates based on user id (assuming user `id` is unique)
      const uniqueResults = Array.from(
        new Set(combinedResults.map((a) => a.id))
      ).map((id) => combinedResults.find((a) => a.id === id));
      //---//

      allUsers = uniqueResults.map((user) => {
        const totalBacklinks = user.backlinks.length;
        const todayBacklinks = user.backlinks.filter((backlink) => {
          const createdAt = new Date(backlink.created_at);
          return createdAt >= todayStart && createdAt <= todayEnd;
        }).length;

        // Count duplicate backlinks: if either 'refering_domain' or 'backlink_duplicate' is true
        const totalDuplicateLinks = user.backlinks.filter((backlink) => {
          return backlink.refering_domain || backlink.backlink_duplicate;
        }).length;

        return {
          ...user,
          totalBacklinks,
          todayBacklinks,
          totalDuplicateLinks,
        };
      });
    } else {
      allUsers = await prismaSubMasterDb.User.findMany({
        where: {
          create_by: existUser.id, // Exclude the current user
          role_id: {
            notIn: [1, 2], // Exclude users with roles 1 and 2
          },
          deleted_at: null,
          email: {
            contains: search_text,
            mode: "insensitive", // Optional: for case-insensitive matching
          },
        },
        orderBy: {
          created_at: "desc",
        },
        skip: skip,
        take: limit,
      });
    }

    // total countss //

    const totalCount = await prismaSubMasterDb.User.count({
      where: {
        deleted_at: null,
        role_id: {
          notIn: [1, 2], // Exclude users with roles 1 and 2
        },
        ...(existUser.role_id === 2 && {
          AND: [
            {
              role_id: {
                notIn: [1, 2],
              },
            },
            {
              OR: [
                { organisation_id: existUser.id },
                { create_by: existUser.id },
              ],
            },
          ],
        }),
        ...(existUser.role_id !== 1 &&
          existUser.role_id !== 2 && {
            create_by: existUser.id,
          }),
      },
    });

    const allRoles = await prismaMasterDb.roles.findMany({
      where: {
        id: { in: allUsers.map((user) => user.role_id) },
        deleted_at: null,
      },
    });

    const userDetails = allUsers.map((user) => {
      const role = allRoles.find((role) => role.id === user.role_id); // Find the corresponding role name
      return {
        ...user,
        roleName: role ? role.name : null,
      };
    });

    const totalPages = Math.ceil(totalCount / limit); // Calculate total pages
    const currentPage = page;

    data = {
      users: userDetails,
      totalCount,
      totalPages,
      currentPage,
      limit,
    };
    return responseHandler.success(res, data, "succes", 200);
  } catch (err) {
    console.log(err);
    return responseHandler.error(
      res,
      err,
      "Error during fetch all user list",
      500
    );
  }
};

// GET TEAM LIST //

const teamList = async (req, res) => {
  let data;
  try {
    const userData = req.user;
    const {
      page_no = 1,
      number_of_rows = 20,
      role_id = "",
      search_text = "",
      user_id = "",
    } = req.query;

    const ROLE = Object.getOwnPropertyNames(userData)[0];

    const existUser = await prismaSubMasterDb.User.findUnique({
      where: {
        id: userData[ROLE].id,
        deleted_at: null,
      },
    });

    if (!existUser) {
      return responseHandler.error(
        res,
        errorCodes.USER_NOT_EXIST,
        errorCodes.USER_NOT_EXIST,
        400
      );
    }

    const page = parseInt(page_no, 10);
    const limit = parseInt(number_of_rows, 10);
    const skip = (page - 1) * limit; // Skip calculation for pagination
    let allUsers;
    let findUser = {};
    if (existUser.role_id == 1) {
      const usersForAdmin = await prismaSubMasterDb.User.findMany({
        where: {
          role_id: {
            in: [2],
          },
          deleted_at: null,
          email: {
            contains: search_text,
            mode: "insensitive", // Optional: for case-insensitive matching
          },
        },
        orderBy: {
          created_at: "desc",
        },
      });
      allUsers = usersForAdmin;
    } else if (existUser.role_id == 2) {
      let excludeRoles = [5];
      if (user_id != "" && user_id != null) {
        findUser = await prismaSubMasterDb.User.findFirst({
          where: {
            id: Number(user_id),
            deleted_at: null,
          },
        });
        if (findUser.role_id === 4) {
          excludeRoles.push(4);
        }
        if (!findUser) {
          return customResponse(res, false, 400, {}, "User not found.");
        }
      }

      const usersByOrganisation = await prismaSubMasterDb.Team.findMany({
        where: {
          organisation_id: existUser.id,
          deleted_at: null,
          UserId: {
            role_id: {
              notIn: excludeRoles,
            },
          },
        },
        orderBy: {
          created_at: "desc", // Order by creation date in descending order
        },
        select: {
          id: true,
          organisation_id: true,
          created_by: true,
          user_id: true,
          role_id: true,
          UserId: {
            select: {
              id: true,
              first_name: true,
              last_name: true,
              email: true,
              role_id: true,
            },
          },
        },
      });

      allUsers = usersByOrganisation.map((item) => {
        const { UserId, ...rest } = item;
        return {
          ...rest,
          user: UserId,
        };
      });
    } else if (existUser.role_id == 3) {
      let excludeRoles = [3, 5];
      if (user_id != "" && user_id != null) {
        findUser = await prismaSubMasterDb.User.findFirst({
          where: {
            id: Number(user_id),
            deleted_at: null,
          },
        });
        if (findUser.role_id === 4) {
          excludeRoles.push(4);
        }
        if (!findUser) {
          return customResponse(res, false, 400, {}, "User not found.");
        }
      }

      // created by manger //
      const allUsersByManger = await prismaSubMasterDb.Team.findMany({
        where: {
          created_by: existUser.id, // Exclude the current user
          deleted_at: null,
          UserId: {
            role_id: {
              notIn: excludeRoles,
            },
          },
        },
        select: {
          id: true,
          organisation_id: true,
          created_by: true,
          user_id: true,
          role_id: true,
          UserId: {
            select: {
              id: true,
              first_name: true,
              last_name: true,
              email: true,
              role_id: true,
            },
          },
        },
        orderBy: {
          created_at: "desc",
        },
      });

      const allUsersAssignToManger =
        await prismaSubMasterDb.AssignTeamMember.findMany({
          where: {
            assign_to_id: existUser.id, // Exclude the current user
            deleted_at: null,
            AssignedUser: {
              role_id: {
                notIn: excludeRoles,
              },
            },
          },
          select: {
            id: true,
            assign_to_id: true,
            assign_by_id: true,
            role_id: true,
            AssignedUser: {
              select: {
                id: true,
                first_name: true,
                last_name: true,
                email: true,
                role_id: true,
                backlinks: true,
              },
            },
          },
          orderBy: {
            created_at: "desc",
          },
        });

      // Normalize and rename the nested user data to `user` for each dataset
      const normalizedByManager = allUsersByManger.map((item) => {
        const { UserId, ...rest } = item;
        return {
          ...rest,
          user: UserId,
        };
      });

      const normalizedAssignedUsers = allUsersAssignToManger.map((item) => {
        const { AssignedUser, ...rest } = item;
        return {
          ...rest,
          user: AssignedUser,
        };
      });

      allUsers = [...normalizedByManager, ...normalizedAssignedUsers];
    } else if (existUser.role_id == 4) {
      let excludeRoles = [3, 5];
      if (user_id != "" && user_id != null) {
        findUser = await prismaSubMasterDb.User.findFirst({
          where: {
            id: Number(user_id),
            deleted_at: null,
          },
        });
        if (findUser.role_id === 4) {
          excludeRoles.push(4);
        }
        if (!findUser) {
          return customResponse(res, false, 400, {}, "User not found.");
        }
      }

      const allUsersByManger = await prismaSubMasterDb.Team.findMany({
        where: {
          created_by: existUser.id, // Exclude the current user
          deleted_at: null,
          UserId: {
            role_id: {
              notIn: excludeRoles,
            },
          },
        },
        select: {
          id: true,
          organisation_id: true,
          created_by: true,
          user_id: true,
          role_id: true,
          UserId: {
            select: {
              id: true,
              first_name: true,
              last_name: true,
              email: true,
              role_id: true,
            },
          },
        },
        orderBy: {
          created_at: "desc",
        },
      });

      const allUsersAssignToManger =
        await prismaSubMasterDb.AssignTeamMember.findMany({
          where: {
            assign_to_id: existUser.id, // Exclude the current user
            deleted_at: null,
            AssignedUser: {
              role_id: {
                notIn: excludeRoles,
              },
            },
          },
          select: {
            id: true,
            assign_to_id: true,
            assign_by_id: true,
            role_id: true,
            AssignedUser: {
              select: {
                id: true,
                first_name: true,
                last_name: true,
                email: true,
                role_id: true,
                backlinks: true,
              },
            },
          },
          orderBy: {
            created_at: "desc",
          },
        });

      // Normalize and rename the nested user data to `user` for each dataset
      const normalizedByManager = allUsersByManger.map((item) => {
        const { UserId, ...rest } = item;
        return {
          ...rest,
          user: UserId,
        };
      });

      const normalizedAssignedUsers = allUsersAssignToManger.map((item) => {
        const { AssignedUser, ...rest } = item;
        return {
          ...rest,
          user: AssignedUser,
        };
      });

      allUsers = [...normalizedByManager, ...normalizedAssignedUsers];
    }
    // allUsers = [...allUsersByLead, ...allUsersAssignToLead];

    data = {
      users: allUsers,
    };
    return responseHandler.success(res, data, "succes", 200);
  } catch (err) {
    console.log(err);
    return responseHandler.error(
      res,
      err,
      "Error during fetch all user list",
      500
    );
  }
};

// MANAGEMENT TEAM LIST//
const fetchManagementTeamList = async (req, res) => {
  let data;
  try {
    const userData = req.user;
    const {
      page_no = 1,
      number_of_rows = 20,
      role_id = "",
      search_text = "",
    } = req.query;

    const ROLE = Object.getOwnPropertyNames(userData)[0];

    const existUser = await prismaSubMasterDb.User.findFirst({
      where: {
        id: userData[ROLE].id,
        deleted_at: null,
      },
      select: {
        id: true,
        email: true,
        UserInfo: {
          select: {
            id: true,
            role_id: true,
            organisation_id: true,
            user_id: true,
          },
        },
      },
    });

    if (!existUser) {
      return customResponse(res, false, 400, {}, "User not found.");
    }

    const page = parseInt(page_no, 10);
    const limit = parseInt(number_of_rows, 10);
    const skip = (page - 1) * limit; // Skip calculation for pagination
    let allUsers = [];
    let totalCounts = 0;

    console.log(existUser, "EXIST USERER");

    if (existUser.UserInfo?.role_id == 1) {
      const usersWithBacklinkCounts = await prismaSubMasterDb.User.findMany({
        where: {
          deleted_at: null,
          UserInfo: {
            role_id: {
              notIn: [1, 2],
            },
          },
          email: {
            contains: search_text,
            mode: "insensitive", // Optional: for case-insensitive matching
          },
        },
        select: {
          id: true,
          email: true,
          first_name: true,
          last_name: true,
          UserInfo: {
            select: {
              id: true,
              user_id: true,
              organisation_id: true,
              role_id: true,
            },
          },
        },
        orderBy: {
          created_at: "desc",
        },

        skip: skip,
        take: limit,
      });

      allUsers = usersWithBacklinkCounts;
    } else if (existUser?.UserInfo?.role_id == 2) {
      const usersByOrganisation = await prismaSubMasterDb.User.findMany({
        where: {
          email: {
            contains: search_text,
            mode: "insensitive",
          },
          deleted_at: null,
          UserInfo: {
            role_id: {
              in: [3, 4, 5],
            },
            organisation_id: existUser?.id,
          },
        },
        orderBy: {
          created_at: "desc",
        },
        select: {
          id: true,
          first_name: true,
          last_name: true,
          email: true,
          UserInfo: {
            select: {
              role_id: true,
              organisation_id: true,
              user_id: true,
            },
          },
        },
        skip: skip,
        take: limit,
      });

      allUsers = usersByOrganisation;

      // totalCounts = await prismaSubMasterDb.User.count({
      //   where: {
      //     email: {
      //       contains: search_text,
      //       mode: "insensitive",
      //     },
      //     id: {
      //       not: existUser.id,
      //     },
      //     role_id: {
      //       in: [3, 4, 5],
      //     },
      //     deleted_at: null,
      //   },
      // });
    } else if (existUser?.UserInfo?.role_id == 3) {
      const allUserForManger = await prismaSubMasterDb.User.findMany({
        where: {
          email: {
            contains: search_text,
            mode: "insensitive", // case-insensitive search
          },

          deleted_at: null,
          UserInfo: {
            is: {
              role_id: {
                in: [3, 4, 5],
              },
              user_id: {
                not: existUser?.id,
              },
              organisation_id: existUser?.UserInfo?.organisation_id,
            },
          },
        },
        orderBy: {
          created_at: "desc",
        },
        select: {
          id: true,
          first_name: true,
          last_name: true,
          email: true,
          UserInfo: {
            select: {
              role_id: true,
              organisation_id: true,
              user_id: true,
            },
          },
        },
        skip: skip,
        take: limit,
      });

      allUsers = allUserForManger;

      //   where: {
      //     email: {
      //       contains: search_text,
      //       mode: "insensitive",
      //     },
      //     id: {
      //       not: existUser.id,
      //     },
      //     role_id: {
      //       in: [3, 4, 5],
      //     },
      //     deleted_at: null,
      //   },
      // });
    } else {
      const allUsersForLead = await prismaSubMasterDb.User.findMany({
        where: {
          email: {
            contains: search_text,
            mode: "insensitive", // case-insensitive search
          },

          deleted_at: null,
          UserInfo: {
            is: {
              role_id: {
                in: [5],
              },
              user_id: {
                not: existUser?.id,
              },
              organisation_id: existUser?.UserInfo?.organisation_id,
            },
          },
        },
        orderBy: {
          created_at: "desc",
        },
        select: {
          id: true,
          first_name: true,
          last_name: true,
          email: true,
          UserInfo: {
            select: {
              role_id: true,
              organisation_id: true,
              user_id: true,
            },
          },
        },
        skip: skip,
        take: limit,
      });

      allUsers = allUsersForLead;

      //   where: {
      //     email: {
      //       contains: search_text,
      //       mode: "insensitive",
      //     },
      //     id: {
      //       not: existUser.id,
      //     },
      //     role_id: {
      //       in: [5],
      //     },
      //     deleted_at: null,
      //   },
      // });
    }

    const allRoles = await prismaMasterDb.roles.findMany({
      where: {
        deleted_at: null,
      },
    });

    const userDetails = allUsers.map((user) => {
      const role = allRoles.find((role) => role.id == user?.UserInfo?.role_id); // Find the corresponding role name

      return {
        ...user,
        roleName: role ? role.name : null,
      };
    });

    // const totalPages = Math.ceil(totalCounts / limit);
    data = {
      users: userDetails,
      // limit: limit,
      // currentPage: Number(page_no),
      // counts: allUsers?.length || 0,
      // totalCounts: totalCounts,
      // totalPages: totalPages,
    };
    return responseHandler.success(res, data, "succes", 200);
  } catch (err) {
    console.log(err);
    return failureResponse(res, err);
  }
};

// USER DETAILS //

const singleUserDetails = async (req, res) => {
  let data;
  try {
    const userData = req.user;
    const { user_id } = req.query;

    const ROLE = Object.getOwnPropertyNames(userData)[0];

    const existUser = await prismaSubMasterDb.User.findUnique({
      where: {
        id: userData[ROLE].id,
        deleted_at: null,
      },
    });

    if (!existUser) {
      return customResponse(res, false, 400, {}, "User not exists.");
    }

    let userDetails = {};
    if (existUser.role_id == 1) {
      const usersWithBacklinkCounts = await prismaSubMasterDb.User.findMany({
        where: {
          role_id: {
            notIn: [1, 2],
          },
          deleted_at: null,
          email: {
            contains: search_text,
            mode: "insensitive", // Optional: for case-insensitive matching
          },
        },
        orderBy: {
          created_at: "desc",
        },
        include: {
          Backlink: {
            where: {
              deleted_at: null,
            },
          },
        },
        skip: skip,
        take: limit,
      });
    } else {
      userDetails = await prismaSubMasterDb.User.findFirst({
        where: {
          id: Number(user_id),
          deleted_at: null,
        },
      });
    }

    data = userDetails;
    return responseHandler.success(res, data, "succes", 200);
  } catch (err) {
    return failureResponse(res, err);
  }
};

// UPDATE USER //

const updateTeam = async (req, res) => {
  let data;
  try {
    const userData = req.user;

    const { user_id, first_name, last_name, email, role_id, assign_to } =
      req.body;
    const ROLE = Object.getOwnPropertyNames(userData)[0];

    const existUser = await prismaSubMasterDb.User.findUnique({
      where: {
        id: userData[ROLE].id,
        deleted_at: null,
      },
    });

    if (!existUser) {
      return customResponse(res, false, 400, {}, "User not exist.");
    }

    const findUser = await prismaSubMasterDb.User.findFirst({
      where: {
        deleted_at: null,
        id: Number(user_id),
      },
    });

    if (!findUser) {
      return customResponse(res, false, 400, {}, "User not exist.");
    }

    const managementExist = await prismaSubMasterDb.User.findFirst({
      where: {
        deleted_at: null,
        id: Number(assign_to),
      },
    });

    let userDetails = {};

    if (existUser.role_id == 1) {
      const usersWithBacklinkCounts = await prismaSubMasterDb.User.findMany({
        where: {
          role_id: {
            notIn: [1, 2],
          },
          deleted_at: null,
          email: {
            contains: search_text,
            mode: "insensitive", // Optional: for case-insensitive matching
          },
        },
        orderBy: {
          created_at: "desc",
        },
        include: {
          Backlink: {
            where: {
              deleted_at: null,
            },
          },
        },
        skip: skip,
        take: limit,
      });
    } else {
      userDetails = await prismaSubMasterDb.User.update({
        where: {
          id: findUser.id,
        },
        data: {
          first_name: first_name,
          last_name: last_name,
        },
      });

      const alreadyCreated = await prismaSubMasterDb.Team.findFirst({
        where: {
          user_id: findUser.id,
          created_by: managementExist.id,
        },
      });

      if (alreadyCreated) {
        return customResponse(
          res,
          false,
          400,
          {},
          "Already member of selected management."
        );
      }

      const alreadyAsssigned =
        await prismaSubMasterDb.AssignTeamMember.findFirst({
          where: {
            deleted_at: null,
            assign_user: findUser.id,
            assign_to_id: managementExist.id,
          },
        });

      if (alreadyAsssigned) {
        return customResponse(
          res,
          false,
          400,
          {},
          "Already member of selected management."
        );
      } else {
        const findAssigned = await prismaSubMasterDb.AssignTeamMember.findFirst(
          {
            where: {
              deleted_at: null,
              assign_user: findUser.id,
              NOT: {
                assign_to_id: managementExist.id,
              },
            },
          }
        );
        if (findAssigned) {
          await prismaSubMasterDb.AssignTeamMember.update({
            where: {
              id: findAssigned.id,
            },
            data: {
              assign_user: findUser.id,
              assign_to_id: managementExist.id,
              assign_by_id: existUser.id,
            },
          });
        } else {
          await prismaSubMasterDb.AssignTeamMember.create({
            data: {
              assign_user: findUser.id,
              assign_to_id: managementExist.id,
              assign_by_id: existUser.id,
              role_id: managementExist?.role_id,
            },
          });
        }
      }
    }

    return customResponse(res, true, 200, {}, "User update successfully.");
  } catch (err) {
    console.log(err);
    return failureResponse(res, err);
  }
};

export {
  registration,
  login,
  verifyEmail,
  forgotPassword,
  resetPassword,
  fetchAllRole,
  addProjectMembers,
  fetchProjectMembers,
  userDetails,
  assignPermission,
  allUserList,
  getAllUserPermissions,
  userPermission,
  myPermission,
  userReports,
  userReportLinks,
  uploadRd,
  userProjects,
  userRd,
  userDetailReport,
  teamList,
  createTeam,
  fetchManagementTeamList,
  singleUserDetails,
  updateTeam,
};
